//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.ee9.websocket.jakarta.tests;

import java.net.URI;
import java.util.concurrent.TimeUnit;

import jakarta.websocket.CloseReason;
import jakarta.websocket.OnOpen;
import jakarta.websocket.Session;
import jakarta.websocket.server.ServerEndpoint;
import org.eclipse.jetty.ee9.servlet.ServletContextHandler;
import org.eclipse.jetty.ee9.websocket.jakarta.client.JakartaWebSocketClientContainer;
import org.eclipse.jetty.ee9.websocket.jakarta.server.JakartaWebSocketServerContainer;
import org.eclipse.jetty.ee9.websocket.jakarta.server.config.JakartaWebSocketServletContainerInitializer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class CloseInOnOpenTest
{
    private Server server;
    private ServerConnector connector;
    private JakartaWebSocketServerContainer serverContainer;
    private JakartaWebSocketClientContainer client;

    @BeforeEach
    public void beforeEach() throws Exception
    {
        server = new Server();

        connector = new ServerConnector(server);
        server.addConnector(connector);

        ServletContextHandler context = new ServletContextHandler();
        context.setContextPath("/");
        server.setHandler(context);

        JakartaWebSocketServletContainerInitializer.configure(context, (servletContext, wsContainer) ->
            wsContainer.addEndpoint(ClosingListener.class));
        server.start();

        serverContainer = JakartaWebSocketServerContainer.getContainer(context.getServletContext());
        assertNotNull(serverContainer);

        client = new JakartaWebSocketClientContainer();
        client.start();
    }

    @AfterEach
    public void afterEach() throws Exception
    {
        client.stop();
        server.stop();
    }

    @Test
    public void testCloseInOnWebSocketOpen() throws Exception
    {
        URI uri = URI.create("ws://localhost:" + connector.getLocalPort() + "/ws");
        EventSocket clientEndpoint = new EventSocket();

        client.connectToServer(clientEndpoint, uri);
        assertTrue(clientEndpoint.closeLatch.await(5, TimeUnit.SECONDS));
        assertThat(clientEndpoint.closeReason.getCloseCode(), is(CloseReason.CloseCodes.VIOLATED_POLICY));

        assertThat(serverContainer.getOpenSessions().size(), is(0));
    }

    @ServerEndpoint("/ws")
    public static class ClosingListener
    {
        @OnOpen
        public void onWebSocketOpen(Session session) throws Exception
        {
            session.close(new CloseReason(CloseReason.CloseCodes.VIOLATED_POLICY, "I am a WS that closes immediately"));
        }
    }
}
