# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the Delay task."""

from django.core.exceptions import ValidationError
from django.utils import timezone

from debusine.server.workflows.models import WorkRequestWorkflowData
from debusine.test.django import TestCase


class DelayTaskTests(TestCase):
    """Test the Delay task."""

    def test_missing_delay_until(self) -> None:
        """The `delay_until` field is required."""
        error_msg = r"delay_until\\n\s+Field required.*type=missing"
        with self.assertRaisesRegex(ValidationError, error_msg):
            self.playground.create_wait_task(
                task_name="delay",
                task_data={},
                workflow_data=WorkRequestWorkflowData(needs_input=False),
            )

    def test_no_additional_properties(self) -> None:
        """Assert no additional properties."""
        error_msg = "Extra inputs are not permitted"
        with self.assertRaisesRegex(ValidationError, error_msg):
            self.playground.create_wait_task(
                task_name="delay",
                task_data={
                    "delay_until": timezone.now().isoformat(),
                    "input": {},
                },
                workflow_data=WorkRequestWorkflowData(needs_input=False),
            )

    def test_label(self) -> None:
        """Test get_label."""
        task = self.playground.create_wait_task(
            task_name="delay",
            task_data={"delay_until": "2024-01-01T12:00:00+00:00"},
            workflow_data=WorkRequestWorkflowData(needs_input=False),
        ).get_task()
        self.assertEqual(
            task.get_label(), "delay until 2024-01-01 12:00:00+00:00"
        )
